//
//  PredictionViewModel.swift
//  death_app Watch App
//
//  ViewModel for prediction display and management
//

import Foundation
import SwiftUI
import Combine

@MainActor
class PredictionViewModel: ObservableObject {
    @Published var lifeExpectancyYears: Double = 0
    @Published var lifeExpectancyDate: Date?
    @Published var changeFromPrevious: String?
    @Published var isImprovement: Bool = false
    @Published var topRiskFactors: [DisplayRiskFactor] = []
    @Published var isLoading = false
    @Published var errorMessage: String?
    
    private let predictionService = PredictionService.shared
    private var cancellables = Set<AnyCancellable>()
    
    init() {
        setupSubscriptions()
        loadInitialPrediction()
    }
    
    func refreshPrediction() {
        Task {
            await predictionService.calculateLifeExpectancy()
        }
    }
    
    func forceRefresh() {
        Task {
            await predictionService.forceRecalculation()
        }
    }
    
    private func setupSubscriptions() {
        // Subscribe to prediction updates
        predictionService.$currentPrediction
            .compactMap { $0 }
            .sink { [weak self] prediction in
                self?.updateDisplay(with: prediction)
            }
            .store(in: &cancellables)
        
        // Subscribe to loading state
        predictionService.$isCalculating
            .assign(to: \.isLoading, on: self)
            .store(in: &cancellables)
        
        // Subscribe to errors
        predictionService.$lastError
            .compactMap { $0 }
            .sink { [weak self] error in
                self?.errorMessage = "Failed to calculate prediction: \(error.localizedDescription)"
            }
            .store(in: &cancellables)
    }
    
    private func loadInitialPrediction() {
        Task {
            await predictionService.calculateLifeExpectancy()
        }
    }
    
    private func updateDisplay(with prediction: LifeExpectancyPrediction) {
        lifeExpectancyYears = prediction.lifeExpectancy
        lifeExpectancyDate = calculateLifeExpectancyDate(from: prediction.lifeExpectancy)
        
        if let change = prediction.changeFromPrevious {
            changeFromPrevious = change.formattedChange
            isImprovement = change.isImprovement
        } else {
            changeFromPrevious = nil
            isImprovement = false
        }
        
        topRiskFactors = convertRiskFactorsForDisplay(prediction.riskFactors)
        errorMessage = nil
    }
    
    private func calculateLifeExpectancyDate(from lifeExpectancy: Double) -> Date {
        let calendar = Calendar.current
        let yearsToAdd = Int(lifeExpectancy)
        let daysToAdd = Int((lifeExpectancy - Double(yearsToAdd)) * 365.25)
        
        var components = DateComponents()
        components.year = yearsToAdd
        components.day = daysToAdd
        
        return calendar.date(byAdding: components, to: Date()) ?? Date()
    }
    
    private func convertRiskFactorsForDisplay(_ riskFactors: RiskFactorBreakdown) -> [DisplayRiskFactor] {
        let factors = [
            ("Heart Rate", riskFactors.heartRate),
            ("Fitness", riskFactors.vo2Max),
            ("Smoking", riskFactors.smoking),
            ("Weight", riskFactors.bmi),
            ("Sleep", riskFactors.sleep)
        ]
        
        return factors
            .filter { abs($0.1 - 1.0) > 0.05 } // Only significant impacts
            .sorted { abs($0.1 - 1.0) > abs($1.1 - 1.0) } // Sort by impact
            .prefix(3) // Top 3
            .map { factor in
                DisplayRiskFactor(
                    name: factor.0,
                    impact: factor.1,
                    description: getRiskFactorDescription(factor.0, impact: factor.1)
                )
            }
    }
    
    private func getRiskFactorDescription(_ name: String, impact: Double) -> String {
        let percentage = abs((impact - 1.0) * 100)
        let direction = impact > 1.0 ? "increases" : "decreases"
        
        return "\(direction) risk by \(Int(percentage))%"
    }
}

// MARK: - Display Models

struct DisplayRiskFactor: Identifiable {
    let id = UUID()
    let name: String
    let impact: Double
    let description: String
    
    var isPositive: Bool { impact < 1.0 }
    var color: Color {
        isPositive ? .green : .red
    }
    
    var iconName: String {
        switch name.lowercased() {
        case "heart rate": return "heart.fill"
        case "fitness": return "figure.run"
        case "smoking": return "smoke.fill"
        case "weight": return "scalemass.fill"
        case "sleep": return "moon.fill"
        default: return "exclamationmark.triangle.fill"
        }
    }
}